"""Simple Tkinter application to batch convert images to WebP format.

The UI is intentionally kept minimal with a classic gray style.
"""

from __future__ import annotations

from dataclasses import dataclass
from pathlib import Path
from typing import Iterable, List

import tkinter as tk
from tkinter import filedialog, messagebox, simpledialog, ttk

try:
    from PIL import Image, ImageTk
except ImportError as exc:  # pragma: no cover - import guard
    raise SystemExit(
        "Pillow is required to run this application. Install it with 'pip install pillow'."
    ) from exc


SUPPORTED_EXTENSIONS = (".jpg", ".jpeg", ".png")
LOGO_FILENAME = "PROJECT X IMAGE OPTIMIZER.PNG"
LOGO_PATH = Path(__file__).with_name(LOGO_FILENAME)


COMPRESSION_PRESETS = {
    "lossless": {"lossless": True, "quality": 100, "method": 6},
    "website": {"lossless": False, "quality": 80, "method": 6},
}


def format_size(num_bytes: int) -> str:
    """Return a human-friendly file size string."""
    if num_bytes <= 0:
        return "0 B"
    units = ["B", "KB", "MB", "GB", "TB"]
    size = float(num_bytes)
    for unit in units:
        if size < 1024 or unit == units[-1]:
            return f"{size:.1f} {unit}"
        size /= 1024


@dataclass
class ConversionResult:
    converted: List[Path]
    skipped: List[Path]


def find_images(input_dir: Path, extensions: Iterable[str]) -> List[Path]:
    """Return a sorted list of image paths within *input_dir* matching *extensions*."""
    images = [
        path
        for path in input_dir.iterdir()
        if path.is_file() and path.suffix.lower() in extensions
    ]
    return sorted(images)


def convert_to_webp(
    image_path: Path,
    output_dir: Path,
    options: dict[str, int | bool],
) -> Path:
    """Convert a single image to WebP format and return the new file path."""
    output_path = output_dir / f"{image_path.stem}.webp"
    with Image.open(image_path) as img:
        save_kwargs = {"format": "WEBP", **options}
        img.save(output_path, **save_kwargs)
    return output_path


def batch_convert(
    input_dir: Path,
    output_dir: Path,
    options: dict[str, int | bool] | None = None,
) -> ConversionResult:
    """Convert all supported images from *input_dir* into WebP files inside *output_dir*."""
    converted: List[Path] = []
    skipped: List[Path] = []

    if options is None:
        options = COMPRESSION_PRESETS["website"]

    images = find_images(input_dir, SUPPORTED_EXTENSIONS)
    for image_path in images:
        try:
            converted_path = convert_to_webp(image_path, output_dir, options)
            converted.append(converted_path)
        except Exception:  # pragma: no cover - surfaced via UI
            skipped.append(image_path)

    return ConversionResult(converted=converted, skipped=skipped)


class WebPConverterApp:
    def __init__(self, root: tk.Tk) -> None:
        self.root = root
        self.root.title("PROJECT X Image Optimizer - Bulk WebP Converter v1.3")
        self.root.configure(bg="#191919")
        self.root.resizable(False, False)
        self.root.attributes("-topmost", True)

        self.input_dir: Path | None = None
        self.output_dir: Path | None = None
        self.logo_image = self._load_logo()
        self.conversion_in_progress = False

        self.current_status_var = tk.StringVar(value="Idle")
        self.total_status_var = tk.StringVar(value="Waiting to start")
        self.compression_var = tk.StringVar(value="website")
        self.results_text: tk.Text | None = None

        self._build_ui()
        self._render_stats([])

    def _build_ui(self) -> None:
        main_bg = "#191919"
        field_bg = "#f0f0f0"
        padding_options = {"padx": 10, "pady": 8, "sticky": "w"}
        has_logo = self.logo_image is not None
        label_col = 0 if not has_logo else 1
        entry_col = label_col + 1
        button_col = entry_col + 1
        row_input = 0
        row_output = 1
        row_formats = 2
        row_options = 3
        row_button = 4
        row_progress = 5
        row_results = 6

        self.root.grid_columnconfigure(entry_col, weight=1)
        self.root.grid_columnconfigure(button_col, weight=0)

        style = ttk.Style()
        try:
            style.theme_use("clam")
        except Exception:
            pass
        style.configure(
            "Dark.Horizontal.TProgressbar",
            troughcolor="#2a2a2a",
            background="#5cb85c",
            bordercolor="#2a2a2a",
            lightcolor="#5cb85c",
            darkcolor="#5cb85c",
        )

        if has_logo:
            logo_label = tk.Label(self.root, image=self.logo_image, bg=main_bg)
            logo_label.grid(row=row_input, column=0, rowspan=3, sticky="nw", padx=(15, 5), pady=(15, 5))

        # Input folder selector
        tk.Label(
            self.root,
            text="Input folder:",
            fg="#ffffff",
            bg=main_bg,
        ).grid(row=row_input, column=label_col, **padding_options)

        self.input_path_var = tk.StringVar(value="No folder selected")
        input_entry = tk.Entry(
            self.root,
            textvariable=self.input_path_var,
            width=45,
            state="readonly",
            readonlybackground=field_bg,
            fg="#000000",
        )
        input_entry.grid(row=row_input, column=entry_col, **padding_options)

        tk.Button(
            self.root,
            text="Browse…",
            command=self._select_input_dir,
            relief=tk.GROOVE,
            bg="#2f2f2f",
            fg="#ffffff",
            activebackground="#3f3f3f",
            activeforeground="#ffffff",
        ).grid(row=row_input, column=button_col, padx=(0, 15), pady=8)

        # Output folder selector
        tk.Label(
            self.root,
            text="Output folder:",
            fg="#ffffff",
            bg=main_bg,
        ).grid(row=row_output, column=label_col, **padding_options)

        self.output_path_var = tk.StringVar(value="No folder selected")
        output_entry = tk.Entry(
            self.root,
            textvariable=self.output_path_var,
            width=45,
            state="readonly",
            readonlybackground=field_bg,
            fg="#000000",
        )
        output_entry.grid(row=row_output, column=entry_col, **padding_options)

        tk.Button(
            self.root,
            text="Browse…",
            command=self._select_output_dir,
            relief=tk.GROOVE,
            bg="#2f2f2f",
            fg="#ffffff",
            activebackground="#3f3f3f",
            activeforeground="#ffffff",
        ).grid(row=row_output, column=button_col, padx=(0, 15), pady=8)

        # Supported formats display (passive)
        tk.Label(
            self.root,
            text="Supported formats:",
            fg="#ffffff",
            bg=main_bg,
        ).grid(row=row_formats, column=label_col, **padding_options)

        supported_formats = ", ".join(ext.upper().lstrip(".") for ext in SUPPORTED_EXTENSIONS)
        tk.Label(
            self.root,
            text=supported_formats,
            bg=field_bg,
            fg="#000000",
            width=30,
            relief=tk.SUNKEN,
        ).grid(row=row_formats, column=entry_col, **padding_options)

        tk.Label(
            self.root,
            text="Compression:",
            fg="#ffffff",
            bg=main_bg,
        ).grid(row=row_options, column=label_col, padx=10, pady=(0, 4), sticky="w")

        options_frame = tk.Frame(self.root, bg=main_bg)
        options_frame.grid(row=row_options, column=entry_col, columnspan=2, sticky="w", pady=(0, 4))

        for value, text in (
            ("lossless", "Lossless"),
            ("website", "Website"),
        ):
            rb = tk.Radiobutton(
                options_frame,
                text=text,
                value=value,
                variable=self.compression_var,
                fg="#ffffff",
                bg=main_bg,
                selectcolor="#2f2f2f",
                activebackground="#2f2f2f",
                activeforeground="#ffffff",
                highlightthickness=0,
            )
            rb.pack(side=tk.LEFT, padx=(0, 12))

        self.convert_button = tk.Button(
            self.root,
            text="Convert to WebP",
            command=self._handle_convert,
            width=20,
            relief=tk.RAISED,
            bg="#3a3a3a",
            fg="#ffffff",
            activebackground="#525252",
            activeforeground="#ffffff",
        )
        self.convert_button.grid(row=row_button, column=label_col, columnspan=3, pady=(15, 10))

        progress_frame = tk.Frame(self.root, bg=main_bg)
        progress_frame.grid(row=row_progress, column=label_col, columnspan=3, padx=10, pady=(0, 15), sticky="we")
        progress_frame.columnconfigure(0, weight=1)

        current_label = tk.Label(
            progress_frame,
            textvariable=self.current_status_var,
            fg="#ffffff",
            bg=main_bg,
            anchor="w",
        )
        current_label.grid(row=0, column=0, sticky="we")

        self.current_progress = ttk.Progressbar(
            progress_frame,
            mode="determinate",
            maximum=1,
            value=0,
            style="Dark.Horizontal.TProgressbar",
        )
        self.current_progress.grid(row=1, column=0, sticky="we", pady=(2, 10))

        total_label = tk.Label(
            progress_frame,
            textvariable=self.total_status_var,
            fg="#ffffff",
            bg=main_bg,
            anchor="w",
        )
        total_label.grid(row=2, column=0, sticky="we")

        self.total_progress = ttk.Progressbar(
            progress_frame,
            mode="determinate",
            maximum=1,
            value=0,
            style="Dark.Horizontal.TProgressbar",
        )
        self.total_progress.grid(row=3, column=0, sticky="we", pady=(2, 0))

        results_label = tk.Label(
            self.root,
            text="File reductions:",
            fg="#ffffff",
            bg=main_bg,
        )
        results_label.grid(row=row_results, column=label_col, padx=10, pady=(0, 4), sticky="nw")

        self.results_text = tk.Text(
            self.root,
            width=60,
            height=6,
            bg="#1f1f1f",
            fg="#ffffff",
            relief=tk.SUNKEN,
            bd=1,
            highlightthickness=0,
            wrap=tk.NONE,
        )
        self.results_text.grid(row=row_results, column=entry_col, columnspan=2, padx=(0, 15), pady=(0, 15), sticky="we")
        self.results_text.configure(state=tk.DISABLED)

    def _render_stats(self, stats: List[tuple[str, int, int, float]]) -> None:
        if self.results_text is None:
            return

        self.results_text.configure(state=tk.NORMAL)
        self.results_text.delete("1.0", tk.END)

        if not stats:
            self.results_text.insert(tk.END, "No files converted yet.\n")
        else:
            for name, original_size, converted_size, reduction in stats:
                if not original_size:
                    reduction_text = "N/A"
                elif reduction >= 0:
                    reduction_text = f"{reduction:.1f}% smaller"
                else:
                    reduction_text = f"{abs(reduction):.1f}% larger"
                self.results_text.insert(
                    tk.END,
                    f"{name}: {reduction_text} ({format_size(original_size)} → {format_size(converted_size)})\n",
                )

        self.results_text.configure(state=tk.DISABLED)

    def _load_logo(self) -> ImageTk.PhotoImage | None:
        if not LOGO_PATH.exists():
            return None
        try:
            image = Image.open(LOGO_PATH)
            max_width = 160
            if image.width > max_width:
                ratio = max_width / image.width
                new_height = max(1, int(image.height * ratio))
                image = image.resize((max_width, new_height), Image.LANCZOS)
            return ImageTk.PhotoImage(image)
        except Exception:
            return None

    def _select_input_dir(self) -> None:
        directory = filedialog.askdirectory(title="Select input folder")
        if directory:
            self.input_dir = Path(directory)
            self.input_path_var.set(str(self.input_dir))

    def _select_output_dir(self) -> None:
        folder_name = simpledialog.askstring(
            "Create output folder",
            "Enter a name for the new output folder:",
            parent=self.root,
        )

        if folder_name is None:
            return

        folder_name = folder_name.strip()
        if not folder_name:
            messagebox.showinfo("Invalid name", "Folder name cannot be empty.")
            return

        directory = filedialog.askdirectory(title="Choose parent folder for output")
        if not directory:
            return

        base_path = Path(directory)

        target_dir = base_path / folder_name
        suffix = 1
        while target_dir.exists():
            target_dir = base_path / f"{folder_name}_{suffix}"
            suffix += 1

        try:
            target_dir.mkdir(parents=True, exist_ok=False)
        except Exception as exc:
            messagebox.showerror("Unable to create folder", f"{exc}")
            return

        self.output_dir = target_dir
        self.output_path_var.set(str(self.output_dir))

        print(f"Output folder set to '{self.output_dir}'.")

    def _validate_selection(self) -> bool:
        if not self.input_dir or not self.input_dir.exists():
            messagebox.showwarning("Missing folder", "Please select a valid input folder.")
            return False
        if not self.output_dir or not self.output_dir.exists():
            messagebox.showwarning("Missing folder", "Please select a valid output folder.")
            return False
        if not any(self.input_dir.iterdir()):
            messagebox.showinfo("No files", "The input folder is empty.")
            return False
        return True

    def _handle_convert(self) -> None:
        if self.conversion_in_progress:
            return

        if not self._validate_selection():
            return

        assert self.input_dir is not None  # for type checkers
        assert self.output_dir is not None

        images = find_images(self.input_dir, SUPPORTED_EXTENSIONS)
        if not images:
            messagebox.showinfo(
                "No supported images",
                "No JPG, JPEG, or PNG files were found in the selected input folder.",
            )
            return

        self.conversion_in_progress = True
        self.convert_button.configure(state=tk.DISABLED)

        self.total_progress.configure(maximum=len(images), value=0)
        self.total_status_var.set(f"0 of {len(images)} completed")
        self.current_progress.configure(maximum=1, value=0)
        self.current_status_var.set("Starting conversion…")

        preset_key = self.compression_var.get()
        preset_options = COMPRESSION_PRESETS.get(preset_key, COMPRESSION_PRESETS["website"]).copy()

        self._render_stats([])
        print(f"Starting conversion of {len(images)} file(s) using preset '{preset_key}'.")
        self._convert_next(images, 0, [], [], preset_options, preset_key, [])

    def _convert_next(
        self,
        images: List[Path],
        index: int,
        converted: List[Path],
        skipped: List[Path],
        options: dict[str, int | bool],
        preset_key: str,
        stats: List[tuple[str, int, int, float]],
    ) -> None:
        if index >= len(images):
            self._finish_conversion(converted, skipped, len(images), stats)
            return

        image_path = images[index]
        self.current_status_var.set(f"Converting {image_path.name}")
        self.current_progress.configure(value=0)
        self.root.update_idletasks()

        print(f"[{index + 1}/{len(images)}] Converting '{image_path.name}' using preset '{preset_key}'.")

        try:
            original_size = image_path.stat().st_size
            converted_path = convert_to_webp(image_path, self.output_dir, options)
            converted_size = converted_path.stat().st_size
            reduction = 0.0
            if original_size:
                reduction = (1 - (converted_size / original_size)) * 100
            stats.append((image_path.name, original_size, converted_size, reduction))
            converted.append(converted_path)
            if original_size:
                change_text = (
                    f"{reduction:.1f}% smaller"
                    if reduction >= 0
                    else f"{abs(reduction):.1f}% larger"
                )
            else:
                change_text = "N/A"
            print(
                f"Successfully saved '{converted_path.name}' ({format_size(original_size)} → {format_size(converted_size)}; {change_text})."
            )
        except Exception as exc:  # pragma: no cover - surfaced via UI
            skipped.append(image_path)
            print(f"Failed to convert {image_path}: {exc}")

        self.current_progress.configure(value=1)
        self.total_progress.configure(value=index + 1)
        self.total_status_var.set(f"{index + 1} of {len(images)} completed")

        self.root.after(
            50,
            lambda: self._convert_next(images, index + 1, converted, skipped, options, preset_key, stats),
        )

    def _finish_conversion(
        self,
        converted: List[Path],
        skipped: List[Path],
        total: int,
        stats: List[tuple[str, int, int, float]],
    ) -> None:
        self.conversion_in_progress = False
        self.convert_button.configure(state=tk.NORMAL)
        self.current_status_var.set("Done")
        self.current_progress.configure(value=0)
        self.total_status_var.set(f"Completed {total} images")

        self._render_stats(stats)

        print("Conversion run complete.")
        if skipped:
            print(f"Skipped {len(skipped)} file(s): {[path.name for path in skipped]}")
        else:
            print("All files converted successfully.")

        summary_lines = [
            f"Converted: {len(converted)}",
            f"Skipped: {len(skipped)}",
            f"Saved to: {self.output_dir}",
        ]

        total_original = sum(item[1] for item in stats)
        total_converted = sum(item[2] for item in stats)
        if total_original:
            overall_reduction = (1 - (total_converted / total_original)) * 100
            overall_text = (
                f"{overall_reduction:.1f}% smaller"
                if overall_reduction >= 0
                else f"{abs(overall_reduction):.1f}% larger"
            )
            summary_lines.append(
                f"Overall change: {overall_text} ({format_size(total_original)} → {format_size(total_converted)})"
            )

        if skipped:
            summary_lines.append("Some files could not be converted. Check the console for details.")

        messagebox.showinfo("Conversion complete", "\n".join(summary_lines))


def main() -> None:
    root = tk.Tk()
    app = WebPConverterApp(root)
    root.mainloop()


if __name__ == "__main__":
    main()

